// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

import java.awt.Color;

/** Represents a material which may have reflective, refractive and specular properties. */
public class Material
{
	/** Material texture. */
	private Texture texture;
	
	/** Specular reflectance value (between 0.0 and 1.0). */
	private float specularReflectance;
	/** Shine value (greater than 0). */
	private float shine;
	
	/** Reflectivity value (between 0.0 and 1.0). */
	private float reflectivity;
	/** Diffuse reflectance value (between 0.0 and 1.0). */
	private float diffuseReflectance;

	/** Transparency value (between 0.0 and 1.0). */
	private float transparency;
	/** Refractive index inside material (greater than 0). */
	private float refractiveIndex;
	
	/** Constructs material.
	 *  @param initTexture             material texture.
	 *  @param initSpecularReflectance specular reflectance value (between 0.0 and 1.0).
	 *  @param initShine               shine value (greater than 0).
	 *  @param initReflectivity        reflectivity value (between 0.0 and 1.0).
	 *  @param initDiffuseReflectance  diffuse reflectance value (between 0.0 and 1.0).
	 *  @param initTransparency        transparency value (between 0.0 and 1.0).
	 *  @param initRefractiveIndex     refractive index inside material (greater than 0).
	 */
	public Material(Texture initTexture, float initSpecularReflectance, float initShine, float initReflectivity, float initDiffuseReflectance, float initTransparency, float initRefractiveIndex)
	{
		specularReflectance = initSpecularReflectance;
		shine               = initShine;
		reflectivity        = initReflectivity;
		diffuseReflectance  = initDiffuseReflectance;
		transparency        = initTransparency;
		refractiveIndex     = initRefractiveIndex;
		texture             = initTexture;
	}

	/** Returns specular reflectance value. */
	public float getSpecularReflectance()
	{
		return specularReflectance;
	}

	/** Returns shine value. */
	public float getShine()
	{
		return shine;
	}

	/** Returns reflectivity value. */
	public float getReflectivity()
	{
		return reflectivity;
	}

	/** Returns diffuse reflectance value. */
	public float getDiffuseReflectance()
	{
		return diffuseReflectance;
	}

	/** Returns transparency value. */
	public float getTransparency()
	{
		return transparency;
	}

	/** Return refective index value. */
	public float getRefractiveIndex()
	{
		return refractiveIndex;
	}

	/** Return whether material is reflective. */
	public boolean isReflective()
	{
		return reflectivity != 0;
	}

	/** Returns whether material is transparent. */
	public boolean isTransparent()
	{
		return transparency != 0;
	}

	/** Returns color at specific position. */
	public Color getColor(float u, float v)
	{
		return texture.getColor(u, v);
	}
}