// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

import java.awt.Color;

/** Represents an infinite plane shape. */
public class Plane extends Shape3d
{
	/** Normal of plane. */
	private Vector3d normal;
	/** Plane position closest to origin. */
	private Vector3d closestPosition;

	/** Constructs a plane.
	 *  @param initMaterial        material plane is made of.
	 *  @param initClosestPosition plane position closest to origin.
	 *  @param initNormal          normal of plane.
	 */
	public Plane(Material initMaterial, Vector3d initClosestPosition, Vector3d initNormal)
	{
		super(initMaterial);

		closestPosition = new Vector3d(initClosestPosition);
		normal          = new Vector3d(initNormal);
	}
	
	public Intersection getIntersection(Ray ray, float threshold)
	{
		// Calculate plane normal components in direction of ray
		float a = ray.getDirection().dot(normal);

		// No intersection if normal and ray are perpendicular
		if (a == 0)
			return null;

		// Calculate vector from ray origin to plane position components in direction of normal
		float b = new Vector3d(ray.getPosition(), closestPosition).dot(normal);
		// Calculate parameter t value where intersection occurs
		float t = b / a;

		// No intersection if t is below threshold
		if (t < threshold)
			return null;
		
		// Return intersection with normal facing towards ray origin
		return new Intersection(this, ray, t, (a > 0) ? normal.reverse() : normal);
	}

	public Color getColor(Vector3d position)
	{
		// TODO: u/v co-ordinates only correct for x/z plane

		// Calculate offset of position from closest point on plane from origin
		Vector3d offset = new Vector3d(position, closestPosition);

		// Return material color at sphere point
		return getMaterial().getColor(offset.x < 0 ? 1 - (-offset.x) % 1 : offset.x, offset.z < 0 ? 1 - (-offset.z) % 1 : offset.z);
	}
}
