// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

import java.awt.event.*;
import java.applet.*;
import java.awt.*;

public class RaytracerApplet extends Applet implements Runnable
{
	/** Thread used for updating applet. */
	private Thread updateThread;
	/** Raytracer component for rendering a scene. */
	private	Raytracer raytracer;

	public void run()
	{
		// Scene to raytrace
		Scene scene = new Scene();

		// Set ambient lighting
		scene.ambientLighting = 0.3f;
		// Set eye ray transformation
		scene.eyeTransform = new Transform3d().rotate(new Vector3d(0.8f, 0, 0)).translate(new Vector3d(0, -3, -3));
		
		// Create light source list
		scene.lightList = new Light[2];
		
		// Add two light sources
		scene.lightList[0] = new Light(new Vector3d( 2, -5, 0), 50.0f);
		scene.lightList[1] = new Light(new Vector3d(-1, -5, 3), 50.0f);

		// Floor and sphere materials
		Material floorMaterial  = new Material(new SolidColorTexture(new Color(0.8f, 0.5f, 0.3f)), 1.0f,  10, 0.3f, 0.4f, 0, 0);
		Material sphereMaterial = new Material(new SolidColorTexture(new Color(0.9f, 0.9f, 1.0f)), 0.8f,  20, 0.7f, 0.9f, 0, 0);

		// Number of sphere to put in a ring
		int numRingSpheres = 9;
		// Radius of sphere ring
		float ringRadius = 3;
		// Radius of spheres
		float sphereRadius = 0.5f;
		
		// Create shape list
		scene.shapeList = new Shape3d[2 + numRingSpheres];
		
		// Add floor plane
		scene.shapeList[0] = new Plane(floorMaterial, new Vector3d(0, 0, 0), new Vector3d(0, -1, 0).normalise());
		// Add sphere in center of scene
		scene.shapeList[1] = new Sphere(sphereMaterial, new Vector3d(0, -sphereRadius , 0), sphereRadius);

		// For each sphere in the ring
		for (int i = 0; i < numRingSpheres ; ++i)
		{
			// Calculate stage of ring
			float stage = (float)((2 * Math.PI) / numRingSpheres) * i;
			// Calculate x and z co-ordinate of sphere in ring
			float x = (float)Math.cos(stage) * ringRadius;
			float z = (float)Math.sin(stage) * ringRadius;

			// Add sphere
			scene.shapeList[i + 2] = new Sphere(sphereMaterial, new Vector3d(x, -sphereRadius , z), sphereRadius);
		}

		// Create raytracer
		raytracer = new Raytracer();
		// Add raytracer component to frame
		add(raytracer);

		// Render scene
		raytracer.render(getGraphics(), scene, getSize().width, getSize().height, true, 10);
	}

	public void start()
	{
		// Create update thread
		updateThread = new Thread(this);
		// Start update thread
		updateThread.start();
	}

	public void stop()
	{
		// Stop update thread
		updateThread = null;
	}

	public void destroy()
	{
		// Stop update thread
		updateThread = null;
	}

	public void paint(Graphics g)
	{
		// If raytracer component has been created
		if (raytracer != null)
		{
			// Paint raytraced scene
			raytracer.paint(g);
		}
	}

	public String getAppletInfo()
	{
		return "Raytracer - Copyright (c) 2001 Sean Wilson. All Rights Reserved.";
	}
}