// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

import java.awt.Color;

/** Represents a sphere shape. */
public class Sphere extends Shape3d
{
	/** Center of sphere. */
	private Vector3d center;
	/** Radius of sphere. */
	private float radius;

	/** North point of sphere for texture mapping. */
	Vector3d northPoint   = new Vector3d(0, 1, 0);
	/** Point on equator of sphere for texture mapping. */
	Vector3d equatorPoint = new Vector3d(1, 0, 0);
	
	/** Constructs a new sphere.
	 *  @param initMaterial material shape is made of.
	 *  @param initCenter   center of sphere.
	 *  @param initRadius   radius of spehre.
	 */
	public Sphere(Material initMaterial, Vector3d initCenter, float initRadius)
	{
		super(initMaterial);

		center = new Vector3d(initCenter);
		radius = initRadius;
	}

	public Intersection getIntersection(Ray ray, float threshold)
	{
		// Calculate vector from ray origin to sphere center
		Vector3d offset = new Vector3d(center, ray.getPosition());
		
		// Calculate terms for sphere intersection
		float b = offset.dot(ray.getDirection()) * 2;
		float c = offset.dot(offset) - radius * radius;
		float d = b * b - 4 * c;

		// Stop if no intersection
		if (d < 0)
			return null;

		// Calculate root of d
		float rootD = (float)Math.sqrt(d);

		// Ray parameter t value at intersection point 
		float t;
		
		// Calculate first intersection point
		float t0 = (-b - rootD) / 2;

		// If intersection point in beyond threshold
		if (t0 > threshold)
		{
			// First intersection point is closest to threshold
			t = t0;
		}
		else
		{
			// Calculate second intersection point
			float t1 = (-b + rootD) / 2;
		
			// No intersection if point is within threshold
			if (t1 <= threshold)
				return null;
			
			// Second intersection point is closest to threshold
			t = t1;
		}

		// Calculate point of intersection
		Vector3d position = ray.getPoint(t);
		// Calculate normal to sphere at intersection point
		Vector3d normal = new Vector3d(center, position).normalise();

		// Return intersection with normal facing towards ray origin
		return new Intersection(this, position, t, (ray.getDirection().dot(normal) > 0) ? normal.reverse() : normal);
	}

	public Color getColor(Vector3d position)
	{
		// Calculate direction of sphere point from sphere center 
		Vector3d direction = new Vector3d(center, position).normalise();
		
		// Calculate angle from north point to sphere point direction
		float northAngle = (float)Math.acos(-northPoint.dot(direction));
		// Calculate angle from equator point to sphere point direction
		float equatorAngle = (float)((Math.acos(direction.dot(equatorPoint) / Math.sin(northAngle)) ) / (2 * Math.PI));
		
		// Calculate u/v co-ordinates
		float u = (northPoint.cross(equatorPoint).dot(direction) < 0) ? equatorAngle : 1 - equatorAngle;
		float v = northAngle / (float)Math.PI;
		
		// Return material color at sphere point
		return getMaterial().getColor(u * radius, v * radius);
	}
}
