// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

/** Represents a 3D vector transformation. */
public class Transform3d
{
	/** Elements of 4 by 3 matrix representing transformation. */ 
	private float element00 = 1, element01 = 0, element02 = 0, element03 = 0,
	              element10 = 0, element11 = 1, element12 = 0, element13 = 0,
	              element20 = 0, element21 = 0, element22 = 1, element23 = 0;
	
	/** Constructs transformation that does nothing. */
	public Transform3d()
	{
	}

	/** Constructs transformation from another.
	 *  @param transform3d transformation to clone.
	 */
	public Transform3d(Transform3d transform3d)
	{
		element00 = transform3d.element00;
		element01 = transform3d.element01;
		element02 = transform3d.element02;
		element03 = transform3d.element03;

		element10 = transform3d.element10;
		element11 = transform3d.element11;
		element12 = transform3d.element12;
		element13 = transform3d.element13;

		element20 = transform3d.element20;
		element21 = transform3d.element21;
		element22 = transform3d.element22;
		element23 = transform3d.element23;
	}

	/** Returns translation transformation added to this transformation.
	 *  @param  delta amount to translate in x, y and z axis.
	 *  @return       resultant transformation.
	 */
	public Transform3d translate(Vector3d delta)
	{
		Transform3d transform3d = new Transform3d();

		transform3d.element03 = delta.x;
		transform3d.element13 = delta.y;
		transform3d.element23 = delta.z;

		return add(transform3d);
	}

	/** Adds scaling transformation to this transformation.
	 *  @param  scaleFactor amount to scale in x, y and z axis.
	 *  @return             resultant transformation.
	 */
	public Transform3d scale(Vector3d scaleFactor)
	{
		Transform3d transform3d = new Transform3d();
		
		transform3d.element00 = scaleFactor.x;
		transform3d.element11 = scaleFactor.y;
		transform3d.element22 = scaleFactor.z;
		
		return add(transform3d);
	}

	/** Returns rotation transformation added to this transformation.
	 *  @param  rotateAmount amount to rotate around x, y and z axis.
	 *  @return              resultant transformation.
	 */
	public Transform3d rotate(Vector3d rotateAmount)
	{
		// Transformation that will rotate around x, y and z axis
		Transform3d fullTransform = new Transform3d();

		// If rotating around x-axis
		if (rotateAmount.x != 0)
		{
			// Calculate sin and cos values
			float sinValue = (float)Math.sin(rotateAmount.x);
			float cosValue = (float)Math.cos(rotateAmount.x);

			// Transformation that will rotate around x-axis
			Transform3d transformX = new Transform3d();
			// Initialise transformation
			transformX.element11 = +cosValue;
			transformX.element12 = +sinValue;
			transformX.element21 = -sinValue;
			transformX.element22 = +cosValue;
			
			// Add transformation to full transformation
			fullTransform = fullTransform.add(transformX);
		}

		// If rotating around y-axis
		if (rotateAmount.y != 0)
		{
			// Calculate sin and cos values
			float sinValue = (float)Math.sin(rotateAmount.y);
			float cosValue = (float)Math.cos(rotateAmount.y);

			// Transformation that will rotate around y-axis
			Transform3d transformY = new Transform3d();
			// Initialise transformation
			transformY.element00 = +cosValue;
			transformY.element02 = +sinValue;
			transformY.element20 = -sinValue;
			transformY.element22 = +cosValue;
			
			// Add transformation to full transformation
			fullTransform = fullTransform.add(transformY);
		}

		// If rotating around z-axis
		if (rotateAmount.z != 0)
		{
			// Calculate sin and cos values
			float sinValue = (float)Math.sin(rotateAmount.z);
			float cosValue = (float)Math.cos(rotateAmount.z);

			// Transformation that will rotate around z-axis
			Transform3d transformZ = new Transform3d();
			// Initialise transformation
			transformZ.element00 = +cosValue;
			transformZ.element01 = +sinValue;
			transformZ.element10 = -sinValue;
			transformZ.element11 = +cosValue;

			// Add transformation to full transformation
			fullTransform = fullTransform.add(transformZ);
		}

		// Return 
		return add(fullTransform);
	}

	/** Returns transformation added to this transformation.
	 *  @param  transform3d transformation to add.
	 *  @return             resultant transformation.
	 */
	public Transform3d add(Transform3d transform3d)
	{
		// Resultant transformation
		Transform3d result = new Transform3d(this);

		// Calculate first column of new transformation matrix
		result.element00 = transform3d.element00 * element00 + transform3d.element01 * element10 + transform3d.element02 * element20;
		result.element01 = transform3d.element00 * element01 + transform3d.element01 * element11 + transform3d.element02 * element21;
		result.element02 = transform3d.element00 * element02 + transform3d.element01 * element12 + transform3d.element02 * element22;
		
		// Calculate second column of new transformation matrix
		result.element03 = transform3d.element00 * element03 + transform3d.element01 * element13 + transform3d.element02 * element23 + transform3d.element03;
		result.element10 = transform3d.element10 * element00 + transform3d.element11 * element10 + transform3d.element12 * element20;
		result.element11 = transform3d.element10 * element01 + transform3d.element11 * element11 + transform3d.element12 * element21;

		// Calculate third column of new transformation matrix
		result.element12 = transform3d.element10 * element02 + transform3d.element11 * element12 + transform3d.element12 * element22;
		result.element13 = transform3d.element10 * element03 + transform3d.element11 * element13 + transform3d.element12 * element23 + transform3d.element13;
		result.element20 = transform3d.element20 * element00 + transform3d.element21 * element10 + transform3d.element22 * element20;

		// Calculate forth column of new transformation matrix
		result.element21 = transform3d.element20 * element01 + transform3d.element21 * element11 + transform3d.element22 * element21;
		result.element22 = transform3d.element20 * element02 + transform3d.element21 * element12 + transform3d.element22 * element22;
		result.element23 = transform3d.element20 * element03 + transform3d.element21 * element13 + transform3d.element22 * element23 + transform3d.element23;

		// Return resultant transformation
		return result;
	}

	/** Transforms a vector.
	 *  @param  transform3d transformation to use.
	 *  @return             transformed vector.
	 */
	public Vector3d transform(Vector3d transform3d)
	{
		// Return vector multiplied by matrix
		return new Vector3d(transform3d.x * element00 + transform3d.y * element01 + transform3d.z * element02 + element03,
		                    transform3d.x * element10 + transform3d.y * element11 + transform3d.z * element12 + element13,
		                    transform3d.x * element20 + transform3d.y * element21 + transform3d.z * element22 + element23);
	}
}