// ====================================================
//
// Copyright (c) 2001 Sean Wilson. All Rights Reserved.
//
// ====================================================

/** Represents a 3D vector. */
public class Vector3d
{
	/** Vector at origin i.e. (0, 0, 0). */
	public static final Vector3d ORIGIN = new Vector3d(0, 0, 0);

	/** Vector x-axis position. */
	public float x = 0;
	/** Vector y-axis position. */
	public float y = 0;
	/** Vector z-axis position. */
	public float z = 0;

	/** Constructs a vector at the origin. */
	public Vector3d()
	{
	}

	/** Constructs a vector.
	 *  @param initX x-axis position.
	 *  @param initY y-axis position.
	 *  @param initZ z-axis position.
	 */
	public Vector3d(float initX, float initY, float initZ)
	{
		x = initX;
		y = initY;
		z = initZ;
	}

	/** Constructs a vector that points from a start to an end position i.e. (end - start).
	 *  @param start start vector.
	 *  @param end   end vector.
	 */
	public Vector3d(Vector3d start, Vector3d end)
	{
		x = end.x - start.x;
		y = end.y - start.y;
		z = end.z - start.z;
	}

	/** Constructs a vector from another.
	 *  @param vector vector to clone.
	 */
	public Vector3d(Vector3d vector)
	{
		x = vector.x;
		y = vector.y;
		z = vector.z;
	}

	/** Returns length of vector. */
	public float getLength()
	{
		return (float)Math.sqrt(x * x + y * y + z * z);
	}

	/** Returns squared length of vector. */
	public float getSquaredLength()
	{
		return x * x + y * y + z * z;
	}

	/** Returns addition of two vectors.
	 *  @param  rhs vector to add.
	 *  @return     (this + rhs).
	 */
	public Vector3d add(Vector3d rhs)
	{
		return new Vector3d(x + rhs.x, y + rhs.y, z + rhs.z);
	}

	/** Returns subtraction of two vectors.
	 *  @param  rhs vector to subtract.
	 *  @return     (this - rhs).
	 */
	public Vector3d subtract(Vector3d rhs)
	{
		return new Vector3d(x - rhs.x, y - rhs.y, z - rhs.z);
	}

	/** Returns multiplication of two vectors.
	 *  @param  rhs vector to multiply by.
	 *  @return     (this * rhs).
	 */
	public Vector3d multiply(Vector3d rhs)
	{
		return new Vector3d(x * rhs.x, y * rhs.y, z * rhs.z);
	}

	/** Returns scaled vector.
	 *  @param  rhs scale factor.
	 *  @return     (this * rhs).
	 */
	public Vector3d multiply(float rhs)
	{
		return new Vector3d(x * rhs, y * rhs, z * rhs);
	}

	/** Returns division of two vector.
	 *  @param  rhs vector to divide by.
	 *  @return     (this / rhs).
	 */
	public Vector3d divide(Vector3d rhs)
	{
		return new Vector3d(x / rhs.x, y / rhs.y, z / rhs.z);
	}

	/** Returns scaled vector.
	 *  @param  rhs scale factor.
	 *  @return     (this / rhs).
	 */
	public Vector3d divide(float rhs)
	{
		return new Vector3d(x / rhs, y / rhs, z / rhs);
	}

	/** Returns vector with direction reversed.
	 *  @return (this * -1).
	 */
	public Vector3d reverse()
	{
		return new Vector3d(-x, -y, -z);
	}

	/** Returns dot product of two vectors.
	 *  @return (this dot rhs).
	 */
	public float dot(Vector3d rhs)
	{
		return x * rhs.x + y * rhs.y + z * rhs.z;
	}

	/** Returns cross product of two vectors.
	 *  @return (this cross rhs).
	 */
	public Vector3d cross(Vector3d rhs)
	{
		return new Vector3d(y * rhs.z - z * rhs.y, z * rhs.x - x * rhs.z, x * rhs.y - y * rhs.x);
	}

	/** Returns normalised vector. */
	public Vector3d normalise()
	{
		float length = getLength();

		return (length == 0 ? ORIGIN : new Vector3d(x / length, y / length, z / length));
	}

	/** Returns vector with same direction but with a specific length.
	 *  @param  length length of new vector.
	 *  @return        resultant vector.
	 */
	public Vector3d resize(float length)
	{
		return normalise().multiply(length);
	}

	/** Returns normal vector of vector from first to second vertex and vector from second to third vertex.
	 *  @param  vertex1 first vertex.
	 *  @param  vertex2 second vertex.
	 *  @param  vertex3 third vertex.
	 *  @return         normal vector.
	 */	
	public static Vector3d getNormal(Vector3d vertex1, Vector3d vertex2, Vector3d vertex3)
	{
		return new Vector3d(vertex2, vertex1).cross(new Vector3d(vertex2, vertex3)).normalise();
	}
	
	/** Returns vector rotated around origin.
	 *  @param rotateAmount amount to rotate around x-axis, y-axis and z-axis.
	 *  @return             rotated vector.
	 */
	public Vector3d rotate(Vector3d rotateAmount)
	{
		// Store old co-ords
		float oldX = x;
		float oldY = y;
		float oldZ = z;

		// Calculate sin and cos of rotation angles
		float sinX = (float)Math.sin(rotateAmount.x);
		float cosX = (float)Math.cos(rotateAmount.x);
		float sinY = (float)Math.sin(rotateAmount.y);
		float cosY = (float)Math.cos(rotateAmount.y);
		float sinZ = (float)Math.sin(rotateAmount.z);
		float cosZ = (float)Math.cos(rotateAmount.z);

		// Rotate around z-axis
		float rotatedX = oldX * cosZ - oldY * sinZ;
		float rotatedY = oldX * sinZ + oldY * cosZ;

		// Store old co-rds
		oldX = rotatedX;
		oldY = rotatedY;

	   	// Rotate around y-axis
		rotatedX = oldX * cosY - oldZ * sinY;
		float rotatedZ = oldX * sinY + oldZ * cosY;

		// Store old co-rds
		oldX = rotatedX;
		oldZ = rotatedZ;

		// Reotate around x-axis
		rotatedY = oldY * cosX - oldZ * sinX;
		rotatedZ = oldY * sinX + oldZ * cosX;

		// Return rotated vector
		return new Vector3d(rotatedX, rotatedY, rotatedZ);
	}

	/** Returns a random unit length vector. */
	public static Vector3d random()
	{
		return new Vector3d((float)Math.random() - 0.5f, (float)Math.random() - 0.5f, (float)Math.random() - 0.5f).normalise();
	}

	public String toString()
	{
		return "(" + x + ", " + y + ", " + z + ")";
	}
}